(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     19896,        698]*)
(*NotebookOutlinePosition[     20982,        733]*)
(*  CellTagsIndexPosition[     20938,        729]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell[TextData["Bras and Kets"], "Title",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData["Colin P. Williams"], "Subsubtitle",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Copyright Notice", "Section",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Copyright Colin P. Williams (1997).

This Notebook is intended to be used in conjunction with \"Explorations in \
Quantum Computing\" by Colin P. Williams and Scott H. Clearwater, TELOS, \
Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to \
copy and distribute this Notebook freely for any non-commercial activity \
provided you include this copyright notice at the beginning of all such \
copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on \"Explorations in Quantum Computing\" check out the TELOS \
web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To \
order call 1-800-777-4643.

All other rights reserved.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData["How to use this Notebook"], "Section",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData[
"This Notebook supplies utilities needed in the quantum computingsimulations. \
These include utilities for working with bras and kets, generating random \
(test) superpositions, extracting amplitudes and probabilities from a ket, \
normalizing kets etc.  Load this Notebook when you run Feynman's quantum \
computer.\n\nThe Notebook contains definitions for:"], "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData[
"KetToColumnVector\nBraToRowVector\nColumnVectorToKet\nRowVectorToBra\n\
KetToBra\nBraToKet\nExpectationValue\nBasisEigenstates\nAmplitudes\n\
Probabilities\nSymbolicCoefficients\nNormalizedKetQ\nNormalizeKet\nDirect\n\
TruthTable\nHermitianQ\nUnitaryQ\nSymbolicSuperposition\n\
RandomSuperposition"], "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Converting Kets to Column Vectors", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
KetToColumnVector[ket[0]]:={{1},{0}}      (* spin up   = 0 *)
KetToColumnVector[ket[1]]:={{0},{1}}      (* spin down = 1 *)
KetToColumnVector[ket[bits__]]:=
\tApply[Direct, Map[KetToColumnVector[ket[#]]&, {bits}]]

KetToColumnVector[a_ ket_ket]:=
\ta KetToColumnVector[ket]
\t
KetToColumnVector[Plus[ket_, kets___]]:=
\tApply[Plus, Map[KetToColumnVector, {ket, kets}]]
\t
KetToColumnVector[superposition_]:=
\tKetToColumnVector[ Expand[superposition] ] \
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Converting Bras to Row Vectors", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
BraToRowVector[bra[0]]:={{1,0}}
BraToRowVector[bra[1]]:={{0,1}}
BraToRowVector[w_. bra[bits__]]:=
\tw * Apply[Direct, Map[BraToRowVector[bra[#]]&, {bits}]]
BraToRowVector[w_. bra[bits__] + bras_.]:=
\tBraToRowVector[w * bra[bits]] + BraToRowVector[bras]
BraToRowVector[superposition_]:=
\tBraToRowVector[Expand[superposition]]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Converting Column Vectors to Kets", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
ColumnVectorToKet[amplitudes_]:=
\tApply[Plus,
\t\t  MapThread[(#1[[1]] #2)&,
\t\t            {amplitudes,
\t\t             EigenKets[ Length[amplitudes] ]
\t\t            }
\t\t           ]
\t\t ]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Converting Row Vectors To Bras", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
RowVectorToBra[{{wi__}}]:=
\tModule[{eigenBras},
\t\teigenBras = EigenKets[Length[{wi}]] /. ket->bra;
\t\tApply[Plus, MapThread[(#1 #2)&, {{wi}, eigenBras}]]
\t]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Converting Between Bras and Kets", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
KetToBra[ket_]:=
\tRowVectorToBra[Conjugate[Transpose[KetToColumnVector[ket]]]]

BraToKet[bra_]:=
\tColumnVectorToKet[Conjugate[Transpose[BraToRowVector[bra]]]]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Average Value of an Observable", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
ExpectationValue[w_. ket[bits__] + kets_., observable_]:=
\t(If[!HermitianQ[observable], 
\t\t(Message[ExpectationValue::notHermitian]; Abort[]),
\t\tIf[Length[observable] != 2^Length[{bits}],
\t\t   (Message[ExpectationValue::incompatible]; Abort[])]];
\t\t       
\t (BraToRowVector[KetToBra[w * ket[bits] + kets]] . 
\t  observable . 
\t  KetToColumnVector[w * ket[bits] + kets]
\t )[[1,1]]  (* scalar = a 1 x 1 matrix, [[1,1]] removes the parentheses *)
\t)

ExpectationValue[superposition_, observable_]:=
\tExpectationValue[Expand[superposition], observable]

ExpectationValue::notHermitian =
\t\"Your purported observable is not an Hermitian matrix.\";
ExpectationValue::incompatible =
\t\"The dimensions of the state vector and observable are incompatible.\";
\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Creating Eigenstates that Span a Hilbert Space", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
BasisEigenstates[m_Integer]:= EigenKets[2^m]

BasisEigenstates::usage = 
  \"BasisEigenstates[m] returns the complete set of \\
  eigenstates that span the Hilbert space of an m-bit \\
  quantum memory register.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
EigenKets[n_]:=
\tModule[{bits},
\t\tbits = Table[Apply[ket, IntegerDigits[i,2]], 
\t\t             {i, 0, n-1}];
\t\t          (* last eigenket has the most bits *)
\t\tMap[PadTo[Length[Last[bits]], #]&, bits]
\t]

PadTo[nDigits_, digits_]:=
\tJoin[Apply[ket, Table[0,{nDigits - Length[digits]}]], 
\t     digits]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["\<\
Accessing Amplitudes of Superpositions and Computing \
Probabilities\
\>", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
Options[Amplitudes] = {ShowEigenstates->False};

ShowEigenstates::usage = 
\t\"ShowEigenstates is an option for Amplitudes that \\
\tdetermines whether the 
output should be a list of the \\
\tamplitudes or a list of {eigenstate, 
amplitude} pairs.\";

Amplitudes[w_. ket[bits__] + kets_., opts___]:=
\tModule[{showeigen},
\tshoweigen = ShowEigenstates /. {opts} /. Options[Amplitudes];
\tWhich[showeigen == True, 
\t\t\tMap[{#, Coefficient[w ket[bits] + kets, #]}&,
\t\t        BasisEigenstates[ Length[{bits}] ]
\t\t       ],
\t\t  showeigen == False,
\t\t    Map[Coefficient[w ket[bits] + kets, #]&,
\t\t        BasisEigenstates[ Length[{bits}] ]
\t\t       ]
\t\t  ]
\t]

(* This clause catches cases like 1/Sqrt[2] (ket[0] + ket[1]) etc *)\t
Amplitudes[c_ (w_. ket[bits__] + kets_.)]:=
\tAmplitudes[ Expand[c (w ket[bits] + kets)] ]

Amplitudes::usage = 
  \"Amplitudes[superposition] returns the amplitudes of the \\
  eigenstates in a superposition or ket vectors.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
General::spell1: 
   Possible spelling error: new symbol name \"Amplitudes\"
     is similar to existing symbol \"amplitudes\".\
\>", "Message",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell["\<\
Options[Probabilities] = {ShowEigenstates->False};

Probabilities[w_. ket[bits__] + kets_., opts___]:=
\tModule[{showeigen, amplitudes, symbols, sumOfSquares},
\t\tshoweigen    = ShowEigenstates /. {opts} /. Options[Probabilities];
\t\tamplitudes   = Amplitudes[w ket[bits] + kets];
\t\tsymbols      = SymbolicCoefficients[amplitudes]; (*see below*)
\t\tsumOfSquares = Simplify[
\t\t                Apply[Plus, 
\t\t                      Map[ComplexExpand[Abs[#]^2, symbols]&, 
\t\t                          amplitudes]]];
\t\tamplitudes   = If[sumOfSquares=!=1,  (* renormalize amplitudes
\t\t                                         if necessary *)
\t\t                  amplitudes/Sqrt[sumOfSquares],
\t\t                  amplitudes];
\t\tWhich[showeigen == True,  
\t\t       MapThread[{#1, ComplexExpand[Abs[#2]^2, symbols]}&, 
\t\t                 {BasisEigenstates[Length[{bits}]], amplitudes}
\t\t                ],
\t\t\t  showeigen == False, 
\t\t\t   Map[ComplexExpand[Abs[#]^2, symbols]&, amplitudes]
\t    ]
\t]

Probabilities[c_ (w_. ket[bits__] + kets_.)]:=
\tProbabilities[ Expand[c (w ket[bits] + kets)] ]

Probabilities::usage =
\t\"Probabilities[superposition] returns the probabilities of \\
\t finding a system in a state described by superposition in \\
\t each of its possible eigenstates upon being measured (observed). \\
\t If Probabilities is given the option ShowEigenstates->True \\
\t the function returns a list of {eigenstate, probability} pairs.\";\
\>", 
  "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
SymbolicCoefficients[amplitudes_List]:=
\tSelect[Union[Flatten[Map[Variables, amplitudes]]], 
\t\t   Not[MatchQ[#, Abs[_]]]&]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Testing Whether a Ket is Properly Normalized", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Needs[\"Algebra`ReIm`\"];

NormalizedKetQ[ket_]:=
\tModule[{columnVector},
\t\tcolumnVector = KetToColumnVector[ket];
\t\t(Inner[Times, 
\t\t       Conjugate[Transpose[columnVector]], 
               columnVector,
               Plus
              ] == {{1}} // N ) /. z_ Conjugate[z_] :> Abs[z]^2
    ]
   
NormalizedKetQ::usage =
\t\"NormalizedKetQ[ket] returns True if the square \\
\tmoduli of the amplitudes of the eigenkets in the \\
\tsuperposition \\\"ket\\\" sum to 1. If \\\"ket\\\" has non-numeric \\
\tamplitudes, the normalization cannot always be determined.\";\
\>", "Input",\

  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["NormalizeKet", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
NormalizeKet[superposition_]:=
\tsuperposition /; NormalizedKetQ[superposition]
NormalizeKet[superposition_]:=
\tExpand[superposition / 
\t       Sqrt[Apply[Plus, 
\t                  Map[Abs[#]^2&, 
\t                      Amplitudes[superposition, 
\t                                 ShowEigenstates->False]
\t                     ]
\t                 ]
\t           ]
\t]
\t      
NormalizeKet::usage =
\t\"NormalizeKet[superposition] is used to normalize a given \\
\tsuperposition of
 ket vectors. That is, if the sum of the squares \\
\tof the absolute values of 
the amplitudes of the eigenstates in \\
\tthe superposition do not sum to 1, 
NormalizeKet rescales the \\
\tamplitudes so that they squared moduli will sum 
to 1.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Direct Product", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
(* Last modified 09/07/96 *)
Needs[\"LinearAlgebra`MatrixManipulation`\"];

Direct[op1_, op2_]:=
\tBlockMatrix[Outer[Times, op1, op2]] /; MatrixQ[op1] && MatrixQ[op2]
\t
Direct[ket_, bra_]:=
\tDirect[KetToColumnVector[ket], BraToRowVector[bra]] /; IsKetQ[ket] && \
IsBraQ[
bra]
\t
Direct[ket1_, ket2_]:=
\tColumnVectorToKet[
\t\tDirect[KetToColumnVector[ket1],
\t           KetToColumnVector[ket2]]
\t]/; IsKetQ[ket1] && IsKetQ[ket2]

Direct[bra1_, bra2_]:=
\tRowVectorToBra[
\t\tDirect[BraToRowVector[bra1],
\t\t\t   BraToRowVector[bra2]]
\t] /; IsBraQ[bra1] && IsBraQ[bra2]
\t
Direct[bra_, ket_]:=
\t(Message[Direct::braket];
\t Direct[BraToRowVector[bra], KetToColumnVector[ket]]) /; IsBraQ[bra] && 
IsKetQ[ket]

Direct[bra_, op_]:=
\t(Message[Direct::braop];
\t Direct[BraToRowVector[bra], op]) /; IsBraQ[bra] && MatrixQ[op]
\t
Direct[op_, bra_]:=
\t(Message[Direct::opbra];
\t Direct[op, BraToRowVector[bra]]) /; MatrixQ[op] && IsBraQ[bra]
\t
Direct[ket_, op_]:=
\t(Message[Direct::ketop];
\t Direct[KetToColumnVector[ket], op]) /; IsKetQ[ket] && MatrixQ[op]
\t
Direct[op_, ket_]:=
\t(Message[Direct::opket];
\t Direct[op, KetToColumnVector[ket]]) /; MatrixQ[op] && IsKetQ[ket]

Direct[matrices__]:=
\tFold[Direct, First[{matrices}], Rest[{matrices}]]

Direct::braket =
\t\"Warning - You are taking the DIRECT product of a bra \\
\tand a ket. This is 
unusual. Perhaps you meant to use \\
\tthe DOT product?\";
\t
Direct::braop =
\t\"Warning - You are taking the DIRECT product of a bra \\
\twith an operator. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";
\t
Direct::opbra =
\t\"Warning - You are taking the DIRECT product of an operator \\
\twith a bra. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";

Direct::ketop =
\t\"Warning - You are taking the DIRECT product of a ket \\
\twith an operator. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";

Direct::opket =
\t\"Warning - You are taking the DIRECT product of an operator \\
\twith a ket. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";


IsKetQ[w_. ket[__] + kets_.]:= True
IsKetQ[_]:=False
\t
IsBraQ[w_. bra[__] + bras_.]:= True
IsBraQ[_]:=False\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Truth Table of a Logic Gate", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
TruthTable[gate_]:=
\tModule[{n,m},
\t\t{n,m} = Dimensions[gate];
\t\tWhich[Not[n==m && IntegerQ[n] && IntegerQ[m]],
\t\t      Message[TruthTable::notsquare]; Abort[],
\t\t      Not[IntegerQ[Log[2, n]]],
\t\t      Message[TruthTable::powerof2]; Abort[]
\t\t     ];
\t\tMap[(# -> ColumnVectorToKet[gate . KetToColumnVector[#]])&, 
\t\t    EigenKets[n]
\t\t   ]  // ColumnForm
\t]
\t
TruthTable::notsquare = 
  \"Your input is not a square matrix and cannot, therefore, represent a \\
  
reversible logic gate.\";

TruthTable::powerof2 = 
  \"Your input is not a matrix of dimensions (2^m) x (2^m) for integer m \\
  
and cannot, therefore, represent a reversible logic gate that operates \\
  on 
m bits.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Types of Operators (Matrices)", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
HermitianQ[matrix_]:=
\tmatrix == Conjugate[Transpose[matrix]]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\
UnitaryQ[matrix_]:=
  Module[{rows, cols},
\t{rows, cols} = Dimensions[matrix];
\tIf[Not[IntegerQ[rows]] || 
\t   Not[IntegerQ[cols]] || 
\t   rows != cols, Message[UnitaryQ::notsquarematrix]];
\t
\t   Chop[Simplify[ComplexExpand[Conjugate[Transpose[matrix]]] - 
\t                 ComplexExpand[Inverse[matrix]]
\t                ]
\t       ] == ZeroMatrix[rows, cols]
  ]

UnitaryQ::notsquarematrix =
  \"Your input is not a square matrix.\";
  
ZeroMatrix[rows_, cols_]:=
\tTable[0, {rows}, {cols}]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Tools for Making Test Superpositions", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SymbolicSuperposition[m_]:=
\tApply[Plus,
\t      MapThread[(#1 #2)&, 
\t                {SymbolicAmplitudes[m], BasisEigenstates[m]}]
\t]
\t
SymbolicSuperposition::usage =
\t\"SymbolicSuperposition[m] creates a superposition of 2^m \\
\teigenstates whose
 amplitudes are uninstantiated symbols. These \\
\teigenstates represent the 
possible states of an m-bit memory \\
\tregister of a quantum computer. This 
function is useful for \\
\texploring the effects of quantum mechanical 
operations on \\
\tarbitrary superpositions. Note that the general form does not

\tguarentee that the superposition is normalized.\";
\t
SymbolicAmplitudes[m_]:=
\t(Clear[w];
\t Map[ToExpression[\"w\"<>ToString[#]]&, Table[i,{i,0,2^m - 1}]]
\t)\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
Options[RandomSuperposition] = {Normalized->True};

RandomSuperposition[m_, opts___]:=
\tModule[{normalized},
\t\tnormalized = Normalized /. {opts} /. Options[RandomSuperposition];
\t\tsuperposition = Apply[Plus,
\t                          MapThread[(#1 #2)&, 
\t                                    {RandomAmplitudes[m],
\t                                     BasisEigenstates[m]}
\t                                   ]
\t                         ];
\t\tWhich[normalized==True, NormalizeKet[superposition],
\t          normalized==False, superposition
\t         ]
\t]

RandomSuperposition::usage =
\t\"RandomSuperposition[m] creates a normalized superposition \\
\tof 2^m eigenstates whose amplitudes are random complex numbers. \\
\tThese eigenstates represent the possible states of an m-bit \\
\tmemory register of a quantum computer. You can generate an \\
\tun-normalized superposition by setting the option Normalized->False.\";
\t
(* You can pick the amplitudes according to whatever distribution
   you like. In the current case we pick random complex numbers
   uniformly from the square in the complex plane bounded by a lower
   left corner at (-1,-I) and an upper right corner at (1,I).
*)
RandomAmplitudes[m_]:=
\tTable[Random[Complex, {-1-I, 1+I}], {2^m}]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
General::spell1: 
   Possible spelling error: new symbol name \"normalized\"
     is similar to existing symbol \"Normalized\".\
\>", "Message",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]]
}, Closed]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"Macintosh 3.0",
ScreenRectangle->{{0, 640}, {0, 460}},
AutoGeneratedPackage->Automatic,
WindowToolbars->{},
CellGrouping->Manual,
WindowSize->{520, 365},
WindowMargins->{{20, Automatic}, {Automatic, 31}},
PrivateNotebookOptions->{"ColorPalette"->{RGBColor, -1}},
ShowCellLabel->True,
ShowCellTags->False,
RenderingOptions->{"ObjectDithering"->True,
"RasterDithering"->False},
MacintoshSystemPageSetup->"\<\
00<0004/0B`000002n88o?mooh<<H`Ui0fL5:0?l0080004/0B`000000]P2:001
0000I00000400`<3000002L?00400@00000000000000060801T1T000000@0000
00000000004000000000000000000000\>"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 88, 2, 98, "Title",
  Evaluatable->False],
Cell[1822, 55, 98, 2, 53, "Subsubtitle",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[1945, 61, 102, 3, 50, "Section",
  Evaluatable->False],
Cell[2050, 66, 905, 19, 222, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[2992, 90, 120, 3, 50, "Section",
  Evaluatable->False],
Cell[3115, 95, 452, 8, 110, "Text",
  Evaluatable->False],
Cell[3570, 105, 370, 7, 259, "Input"],

Cell[CellGroupData[{
Cell[3965, 116, 103, 2, 46, "Subsection",
  Evaluatable->False],
Cell[4071, 120, 532, 16, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[4640, 141, 100, 2, 30, "Subsection",
  Evaluatable->False],
Cell[4743, 145, 405, 11, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[5185, 161, 103, 2, 30, "Subsection",
  Evaluatable->False],
Cell[5291, 165, 276, 11, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[5604, 181, 100, 2, 30, "Subsection",
  Evaluatable->False],
Cell[5707, 185, 240, 8, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[5984, 198, 102, 2, 30, "Subsection",
  Evaluatable->False],
Cell[6089, 202, 239, 8, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[6365, 215, 100, 2, 30, "Subsection",
  Evaluatable->False],
Cell[6468, 219, 847, 23, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[7352, 247, 116, 2, 30, "Subsection",
  Evaluatable->False],
Cell[7471, 251, 293, 9, 70, "Input",
  InitializationCell->True],
Cell[7767, 262, 393, 14, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[8197, 281, 146, 5, 30, "Subsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[8368, 290, 1053, 33, 70, "Input",
  InitializationCell->True],
Cell[9424, 325, 202, 6, 70, "Message",
  Evaluatable->False]
}, Open  ]],
Cell[9641, 334, 1564, 37, 70, "Input",
  InitializationCell->True],
Cell[11208, 373, 204, 6, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[11449, 384, 114, 2, 30, "Subsection",
  Evaluatable->False],
Cell[11566, 388, 656, 21, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[12259, 414, 82, 2, 30, "Subsection",
  Evaluatable->False],
Cell[12344, 418, 815, 26, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[13196, 449, 84, 2, 30, "Subsection",
  Evaluatable->False],
Cell[13283, 453, 2269, 86, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[15589, 544, 97, 2, 30, "Subsection",
  Evaluatable->False],
Cell[15689, 548, 785, 27, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[16511, 580, 99, 2, 30, "Subsection",
  Evaluatable->False],
Cell[16613, 584, 113, 4, 38, "Input"],
Cell[16729, 590, 580, 21, 272, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[17346, 616, 106, 2, 30, "Subsection",
  Evaluatable->False],
Cell[17455, 620, 809, 27, 70, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[18289, 651, 1350, 33, 70, "Input",
  InitializationCell->True],
Cell[19642, 686, 202, 6, 70, "Message",
  Evaluatable->False]
}, Open  ]]
}, Closed]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

